document.addEventListener('DOMContentLoaded', function() {

    // --- Mobile Menu Logic ---
    const mobileMenuToggle = document.getElementById('mobile-menu-toggle');
    const mobileMenu = document.getElementById('mobile-menu');
    const body = document.body;

    if (mobileMenuToggle && mobileMenu) {
        mobileMenuToggle.addEventListener('click', () => {
            mobileMenuToggle.classList.toggle('active');
            mobileMenu.classList.toggle('active');
            body.classList.toggle('mobile-menu-open'); // Toggle body scroll lock
        });
    }

    // --- Downloader Box Logic ---
    const downloaderBoxes = document.querySelectorAll('.ts-downloader-box');
    const resultsDivGlobal = document.querySelector('#ts-results'); 

    // Early exit if essential data is missing
    if (typeof tsDownloaderData === 'undefined' || !tsDownloaderData.allTexts) {
        console.error('ToolsSuite Error: Essential localization data (tsDownloaderData) not found in HTML.');
        if (resultsDivGlobal) {
            resultsDivGlobal.classList.remove('ts-results-hidden');
            resultsDivGlobal.innerHTML = '<div class="ts-message ts-error">A critical configuration error occurred. Please contact support.</div>';
        }
        return;
    }
    
    const texts = tsDownloaderData.allTexts;

    // --- Global Helper Functions ---
    function escapeHtml(unsafe) {
        if (unsafe === null || typeof unsafe === 'undefined') return '';
        return unsafe.toString().replace(/&/g, "&amp;").replace(/</g, "&lt;").replace(/>/g, "&gt;").replace(/"/g, "&quot;").replace(/'/g, "&#039;");
    }

    function displayGlobalMessage(type, messageContent) {
        if (!resultsDivGlobal) return;
        resultsDivGlobal.classList.remove('ts-results-hidden');
        resultsDivGlobal.innerHTML = `<div class="ts-message ts-${type}">${messageContent}</div>`;
        resultsDivGlobal.scrollIntoView({ behavior: 'smooth', block: 'center' });
    }
    
    function setupModal(openBtnId, modalId, resultsContainer) {
        const triggerContainer = resultsContainer || document;
        const openModalBtn = triggerContainer.querySelector(`#${openBtnId}`);
        const modalOverlay = document.querySelector(`#${modalId}`);

        if (openModalBtn && modalOverlay) {
            const closeModalBtn = modalOverlay.querySelector('.ts-modal-close-btn');
            const modalTitle = modalOverlay.querySelector('.ts-modal-title');
            const modalBody = modalOverlay.querySelector('.ts-modal-body');

            // Populate modal content from texts object
            if (modalId === 'ts-bookmark-modal' && modalTitle && modalBody) {
                modalTitle.innerHTML = escapeHtml(texts.modal_bookmark_title);
                modalBody.innerHTML = `
                    <p>${texts.modal_bookmark_intro}</p>
                    <h5>${texts.modal_bookmark_mobile_title}</h5>
                    <p>${texts.modal_bookmark_mobile_para}</p>
                    <h5>${texts.modal_bookmark_desktop_title}</h5>
                    <ul>
                        <li>${texts.modal_bookmark_desktop_chrome}</li>
                        <li>${texts.modal_bookmark_desktop_firefox}</li>
                        <li>${texts.modal_bookmark_desktop_edge}</li>
                        <li>${texts.modal_bookmark_desktop_safari}</li>
                    </ul>`;
            } else if (modalId === 'ts-donation-modal' && modalTitle && modalBody) {
                 modalTitle.innerHTML = escapeHtml(texts.modal_support_title);
                 modalBody.innerHTML = `
                    <p>${escapeHtml(texts.modal_support_intro)}</p>
                    <div class="ts-donation-address"><strong>${escapeHtml(texts.wallet_btc_label)}:</strong><span>${escapeHtml(texts.wallet_btc_address)}</span></div>
                    <div class="ts-donation-address"><strong>${escapeHtml(texts.wallet_usdt_label)}:</strong><span>${escapeHtml(texts.wallet_usdt_address)}</span></div>
                    <div class="ts-donation-address"><strong>${escapeHtml(texts.wallet_eth_label)}:</strong><span>${escapeHtml(texts.wallet_eth_address)}</span></div>
                    <div class="ts-donation-address"><strong>${escapeHtml(texts.wallet_trx_label)}:</strong><span>${escapeHtml(texts.wallet_trx_address)}</span></div>
                    <div class="ts-donation-address"><strong>${escapeHtml(texts.wallet_sol_label)}:</strong><span>${escapeHtml(texts.wallet_sol_address)}</span></div>
                    <p style="margin-top: 15px; font-size: 0.85em; color: #6c757d;">${escapeHtml(texts.modal_support_disclaimer)}</p>
                    <p>${escapeHtml(texts.modal_support_thanks)}</p>`;
            }

            const openHandler = (e) => {
                e.preventDefault();
                modalOverlay.classList.add('active');
            };
            const closeHandler = (e) => {
                e.preventDefault();
                modalOverlay.classList.remove('active');
            };
            
            openModalBtn.addEventListener('click', openHandler);
            if (closeModalBtn) closeModalBtn.addEventListener('click', closeHandler);
            modalOverlay.addEventListener('click', (e) => {
                if (e.target === modalOverlay) modalOverlay.classList.remove('active');
            });
        }
    }

    function displaySuccessResult(apiResponse) {
        const { link: downloadUrl, file_name: fileName } = apiResponse;
        const fileExtension = fileName.split('.').pop().toLowerCase();
        
        let landerUrl = new URL(tsDownloaderData.siteUrl + 'download/', window.location.origin);
        landerUrl.searchParams.append('file_url', btoa(downloadUrl));
        landerUrl.searchParams.append('file_name', btoa(fileName));

        const getLinkButtonText = texts.ui_get_link_button_text;
        const siteUrl = tsDownloaderData.siteUrl;
        const shareText = texts.text_share_default;
        const encodedSiteUrl = encodeURIComponent("https://telegramdownloader.net/");
        const encodedShareText = encodeURIComponent(shareText);
        const pinterestImage = encodeURIComponent("https://telegramdownloader.net/img/social-share.jpg");

        const successHTML = `
            <div class="ts-file-info">
                <p class="ts-file-name"><strong>File Name:</strong> ${escapeHtml(fileName)}</p>
                <p class="ts-file-format"><strong>Format:</strong> ${escapeHtml(fileExtension.toUpperCase())}</p>
            </div>
            <div class="ts-actions-area">
                <a href="${landerUrl.href}" target="_blank" rel="nofollow noopener" class="ts-action-button ts-download-link-button">
                    <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16"><path d="M8 0a8 8 0 1 1 0 16A8 8 0 0 1 8 0zM4.5 7.5a.5.5 0 0 0 0 1h5.793l-2.147 2.146a.5.5 0 0 0 .708.708l3-3a.5.5 0 0 0 0-.708l-3-3a.5.5 0 1 0-.708.708L10.293 7.5H4.5z"/></svg>
                    ${escapeHtml(getLinkButtonText)}
                </a>
            </div>
            <div class="ts-extra-sections">
                <div class="ts-bookmark-section">
                    <h3 class="ts-section-title">${escapeHtml(texts.section_bookmark_title)}</h3>
                    <p>${texts.section_bookmark_paragraph}</p>
                    <button type="button" class="ts-extra-button" id="ts-open-bookmark-modal-btn">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16"><path fill-rule="evenodd" d="M2 15.5V2a2 2 0 0 1 2-2h8a2 2 0 0 1 2 2v13.5a.5.5 0 0 1-.74.439L8 13.069l-5.26 2.87A.5.5 0 0 1 2 15.5zm6.5-11a.5.5 0 0 0-1 0V6H6a.5.5 0 0 0 0 1h1.5v1.5a.5.5 0 0 0 1 0V7H10a.5.5 0 0 0 0-1H8.5V4.5z"/></svg>
                        ${escapeHtml(texts.button_bookmark_text)}
                    </button>
                </div>
                <div class="ts-share-section">
                    <h3 class="ts-section-title">${escapeHtml(texts.section_share_title)}</h3>
                    <p>${texts.section_share_paragraph}</p>
                    <div class="ts-share-buttons">
                        <a href="https://twitter.com/intent/tweet?text=${encodedShareText}" target="_blank" class="ts-share-button" title="Share on X"><svg viewBox="0 0 24 24" aria-hidden="true"><g><path d="M18.244 2.25h3.308l-7.227 8.26 8.502 11.24H16.17l-5.214-6.817L4.99 21.75H1.68l7.73-8.835L1.254 2.25H8.08l4.713 6.231zm-1.161 17.52h1.833L7.084 4.126H5.117z"></path></g></svg></a>
                        <a href="https://www.facebook.com/sharer/sharer.php?u=${encodedSiteUrl}&quote=${encodedShareText}" target="_blank" class="ts-share-button" title="Share on Facebook"><svg viewBox="0 0 24 24"><path d="M14 13.5h2.5l1-4H14v-2c0-1.03 0-2 2-2h1.5V2.14c-.326-.043-1.557-.14-2.857-.14C11.928 2 10 3.657 10 6.7v2.8H7v4h3V22h4v-8.5z"></path></svg></a>
                        <a href="https://api.whatsapp.com/send?text=${encodedShareText}" target="_blank" class="ts-share-button" title="Share on WhatsApp"><svg viewBox="0 0 24 24"><path d="M19.05 4.94a10 10 0 0 0-14.14 0 10 10 0 0 0 0 14.14 10 10 0 0 0 14.14 0 10 10 0 0 0 0-14.14zM12 20.16a8.16 8.16 0 1 1 0-16.32 8.16 8.16 0 0 1 0 16.32zm4.5-5.31c-.26-.13-1.55-.77-1.79-.86-.24-.09-.41-.13-.59.13-.17.26-.68.85-.83 1.02-.15.17-.31.19-.58.07-.27-.13-1.16-.42-2.2-1.36-.82-.73-1.37-1.62-1.52-1.89-.15-.26 0-.4.12-.54.12-.12.26-.31.39-.46.13-.15.17-.26.26-.44.09-.17.04-.31-.02-.44-.06-.13-.58-1.41-.8-1.93s-.41-.44-.58-.45c-.16 0-.33 0-.5 0s-.44.06-.68.33c-.23.26-.9.88-1.02 2.12.12 1.24.92 2.45 1.05 2.62.13.17 1.76 2.67 4.27 3.77.6.26 1.11.42 1.49.53.68.21 1.3.18 1.78.11.54-.08 1.55-.63 1.77-1.24.22-.61.22-1.13.15-1.24-.06-.12-.24-.17-.5-.31z"/></svg></a>
                        <a href="https://t.me/share/url?url=${encodedSiteUrl}&text=${encodedShareText}" target="_blank" class="ts-share-button" title="Share on Telegram"><svg viewBox="0 0 24 24"><path d="m9.417 15.181-.397 5.584c.568 0 .814-.244 1.109-.537l2.663-2.545 5.518 4.041c1.012.564 1.725.267 1.998-.931L22.83 3.446c.34-1.626-.57-2.34-1.574-1.945L1.442 10.183c-1.63.62-1.63 1.52.285 1.88l5.232 1.634 12.31-7.742c.576-.354 1.12.112.652.502l-9.94 8.697z"/></svg></a>
                    </div>
                </div>
                <div class="ts-financial-support-section">
                     <h3 class="ts-section-title">${escapeHtml(texts.section_support_title)}</h3>
                     <p>${texts.section_support_paragraph}</p>
                     <button type="button" class="ts-support-button" id="ts-open-donation-modal-btn">
                        <svg xmlns="http://www.w3.org/2000/svg" width="16" height="16" fill="currentColor" viewBox="0 0 16 16"><path fill-rule="evenodd" d="M8 1.314C12.438-3.248 23.534 4.735 8 15-7.534 4.736 3.562-3.248 8 1.314z"/></svg>
                        ${escapeHtml(texts.button_support_text)}
                    </button>
                </div>
            </div>
        `;
        displayGlobalMessage('success', successHTML);
        
        const successMessageContainer = resultsDivGlobal.querySelector('.ts-message.ts-success');
        if (successMessageContainer) {
            setupModal('ts-open-bookmark-modal-btn', 'ts-bookmark-modal', successMessageContainer);
            setupModal('ts-open-donation-modal-btn', 'ts-donation-modal', successMessageContainer);
        }
    }

    downloaderBoxes.forEach(downloadBox => {
        const linkInput = downloadBox.querySelector('#ts-telegram-link');
        const pasteButton = downloadBox.querySelector('#ts-paste-btn');
        const downloadButton = downloadBox.querySelector('#ts-download-btn');

        if (!linkInput || !pasteButton || !downloadButton) return;

        function updatePasteButtonState() {
            pasteButton.textContent = (linkInput.value.trim() !== '') ? texts.ui_clear_button_text : texts.ui_paste_button_text;
        }
        
        pasteButton.addEventListener('click', function() {
            if (resultsDivGlobal) { 
                resultsDivGlobal.innerHTML = '';
                resultsDivGlobal.classList.add('ts-results-hidden');
            }
            if (pasteButton.textContent === texts.ui_paste_button_text) {
                if (navigator.clipboard && navigator.clipboard.readText) {
                    navigator.clipboard.readText().then(text => {
                        linkInput.value = text.trim();
                        updatePasteButtonState();
                    }).catch(err => {
                        displayGlobalMessage('error', texts.error_clipboard_read);
                    });
                } else {
                    displayGlobalMessage('error', texts.error_clipboard_unavailable);
                }
            } else {
                linkInput.value = '';
                updatePasteButtonState();
                linkInput.focus();
            }
        });

        linkInput.addEventListener('input', updatePasteButtonState);
        updatePasteButtonState(); 

        downloadButton.addEventListener('click', function() {
            if (!resultsDivGlobal) {
                alert(texts.error_missing_results_container);
                return;
            }
            const telegramLink = linkInput.value.trim();
            if (!telegramLink) {
                displayGlobalMessage('error', texts.error_empty_link);
                linkInput.focus();
                return;
            }
            
            const privateLinkRegex = /\/t\.me\/c\//;
            if (privateLinkRegex.test(telegramLink)) {
                displayGlobalMessage('error', texts.api_406_chat_forwards_restricted);
                return;
            }

            const TEN_MINUTES = 10 * 60 * 1000;
            const now = new Date().getTime();

            try {
                const cachedErrorString = sessionStorage.getItem('ts_error_cache_' + telegramLink);
                if (cachedErrorString) {
                    const cachedError = JSON.parse(cachedErrorString);
                    if ((now - cachedError.timestamp) < TEN_MINUTES && cachedError.message) {
                        displayGlobalMessage('loading', texts.ui_loading_text);
                        setTimeout(() => { displayGlobalMessage('error', escapeHtml(cachedError.message)); }, 500);
                        return;
                    }
                }
            } catch (e) {
                console.warn("Could not read error from session storage.", e);
            }

            try {
                const cachedResultString = sessionStorage.getItem('ts_cache_' + telegramLink);
                if (cachedResultString) {
                    const cachedResult = JSON.parse(cachedResultString);
                    if ((now - cachedResult.timestamp) < TEN_MINUTES && cachedResult.data && cachedResult.data.link) {
                        displayGlobalMessage('loading', texts.ui_loading_text);
                        setTimeout(() => { displaySuccessResult(cachedResult.data); }, 500);
                        return; 
                    }
                }
            } catch (e) {
                console.warn("Could not read success from session storage.", e);
            }

            const originalButtonText = downloadButton.textContent; 
            displayGlobalMessage('loading', texts.ui_loading_text);
            downloadButton.disabled = true;
            downloadButton.textContent = texts.ui_loading_text;

            const formData = new URLSearchParams();
            formData.append('telegram_link', telegramLink);

            fetch(tsDownloaderData.ajaxUrl, {
                method: 'POST',
                headers: { 'Content-Type': 'application/x-www-form-urlencoded' },
                body: formData.toString()
            })
            .then(response => {
                if (!response.ok) {
                    return response.json().then(err => {
                        err.statusCode = response.status;
                        throw err;
                    });
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    const apiResponse = data.data.data;
                    const apiStatusCode = data.data.api_status_code;

                    if (apiStatusCode >= 200 && apiStatusCode < 300 && apiResponse && apiResponse.status === "success" && apiResponse.link && apiResponse.file_name) {
                        try {
                            const cacheObject = { data: apiResponse, timestamp: new Date().getTime() };
                            sessionStorage.setItem('ts_cache_' + telegramLink, JSON.stringify(cacheObject));
                        } catch (e) {
                            console.warn("Could not write to session storage.", e);
                        }
                        displaySuccessResult(apiResponse);
                    } else {
                        handleApiError(apiStatusCode, apiResponse, telegramLink);
                    }
                } else {
                     const errorData = data.data || {};
                     const apiStatusCode = errorData.api_status_code || null;
                     handleApiError(apiStatusCode, errorData.data || {message: texts.api_unknown_error }, telegramLink);
                }
            })
            .catch(error => {
                console.error('Fetch/Processing Error:', error);
                 const apiStatusCode = error.statusCode || null;
                 const errorData = error.data || { message: texts.error_network };
                 handleApiError(apiStatusCode, errorData, telegramLink);
            })
            .finally(() => {
                downloadButton.disabled = false;
                downloadButton.textContent = originalButtonText; 
            });
        });
    }); 

    // --- YouTube Lazy Load ---
    const youtubePlayers = document.querySelectorAll('.youtube-player');
    youtubePlayers.forEach(player => {
        player.addEventListener('click', () => {
            const iframe = document.createElement('iframe');
            iframe.setAttribute('frameborder', '0');
            iframe.setAttribute('allowfullscreen', '');
            iframe.setAttribute('allow', 'autoplay; encrypted-media');
            iframe.setAttribute('src', `https://www.youtube.com/embed/${player.dataset.id}?autoplay=1&rel=0`);
            player.innerHTML = '';
            player.appendChild(iframe);
        }, { once: true });
    });
});

